/**
  ********************************************************************************
  *
  * @file    hal_gpio.h
  * @brief   Header file of Gpio driver
  *
  * @version V1.0
  * @data    22 Aug 2018
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  ********************************************************************************
  */


#ifndef __HAL_GPIO_H__
#define __HAL_GPIO_H__


#include <hic.h>
#include "type.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @addtogroup Gpio
  * @{
  */

/**
  * @defgroup Gpio_Public_Types gpio Public Types
  * @{
  */

/**
  * @brief Type of gpio
  */
typedef enum {
	PA_TYPE = 0x0,
	PB_TYPE = 0x1,
	PE_TYPE = 0x2,
} gpio_type_t;

/**
  * @brief Number of pin
  */
typedef enum {
	GPIO_PIN_0   = 0x0,
	GPIO_PIN_1   = 0x1,
	GPIO_PIN_2   = 0x2,
	GPIO_PIN_3   = 0x3,
	GPIO_PIN_4   = 0x4,
	GPIO_PIN_5   = 0x5,
	GPIO_PIN_6   = 0x6,
	GPIO_PIN_7   = 0x7,
} gpio_pin_t;

/**
  * @brief Select gpio direction
  */
typedef enum {
	GPIO_OUTPUT = 0x0,
	GPIO_INPUT  = 0x1,
} gpio_io_t;

/**
  * @brief Select gpio initialization level
  */
typedef enum {
	GPIO_LOW_LEVEL  = 0x0,
	GPIO_HIGH_LEVEL = 0x1,
} gpio_level_t;

/**
  * @brief Select gpio digit or analog
  */
typedef enum {
	GPIO_DIGIT_MODE  = 0x0,
	GPIO_ANALOG_MODE = 0x1,
} gpio_mode_t;

/**
  * @brief Select gpio driver ability
  */
typedef enum {
	GPIO_LOW_DRIVER  = 0x0,
	GPIO_HIGH_DRIVER = 0x1,
} gpio_drive_t;

/**
  * @brief Select gpio smit
  */
typedef enum {
	GPIO_1_5_TO_3_0_SMIT = 0,
	GPIO_0_8_TO_2_0_SMIT = 1,
} type_smit_t;

/**
  * @brief Select gpio curruct source
  */
typedef enum {
	GPIO_SC_4_MA     = 0,
	GPIO_SC_7_MA     = 1,
	GPIO_SC_11_MA    = 2,
	GPIO_SC_HIGH_REG = 3,
} type_sc_t;

/**
  * @brief  GPIO handle Structure definition
  */
typedef struct gpio_init_s {
	gpio_io_t io;
	gpio_level_t level;
	gpio_mode_t mode;
	type_func_t pu;
	type_func_t pd;
	gpio_drive_t drive;
	type_func_t od;
	type_smit_t smit;
	type_func_t cur_ctr;
	type_sc_t src_cur;
} gpio_init_t;
/**
  * @}
  */

/**
  * @defgroup GPIO_Private_Micros gpio private Micros
  * @{
  */	
#define IS_GPIO_TYPE(x)			(((x) == PA_TYPE) || \
					 ((x) == PB_TYPE) || \
					 ((x) == PE_TYPE))
#define IS_GPIO_IO_TYPE(x)		(((x) == GPIO_INPUT) || \
					 ((x) == GPIO_OUTPUT))
#define IS_GPIO_LEVEL_TYPE(x)		(((x) == GPIO_LOW_LEVEL) || \
					 ((x) == GPIO_HIGH_LEVEL))
#define IS_GPIO_MODE_TYPE(x)		(((x) == GPIO_DIGIT_MODE) || \
					 ((x) == GPIO_ANALOG_MODE))
#define IS_GPIO_DRIVER_TYPE(x)		(((x) == GPIO_LOW_DRIVER) || \
					 ((x) == GPIO_HIGH_DRIVER))
#define IS_GPIO_PIN_TYPE(x)		((x) < 9)
#define IS_GPIO_SMIT_TYPE(x)		(((x) == GPIO_1_5_TO_3_0_SMIT) || \
					 ((x) == GPIO_0_8_TO_2_0_SMIT))
#define IS_GPIO_SC_TYPE(x)		(((x) == GPIO_SC_4_MA)  || \
					 ((x) == GPIO_SC_7_MA)  || \
					 ((x) == GPIO_SC_11_MA) || \
					 ((x) == GPIO_SC_HIGH_REG))
/**
  * @}
  */

/**
  * @addtogroup GPIO_Public_Functions
  * @{
  */
						 
hal_status_t gpio_init(gpio_type_t gpio, gpio_pin_t pin, gpio_init_t *init);
void gpio_write_pin(gpio_type_t gpio, gpio_pin_t pin, uint8_t val);
uint8_t gpio_read_pin(gpio_type_t gpio, gpio_pin_t pin);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#endif /* __HAL_GPIO_H */
